// Smooth Page Transitions
(function() {
    'use strict';

    // Prevent multiple transitions
    let isTransitioning = false;

    // Add fade-in animation on page load
    function fadeIn() {
        document.body.style.opacity = '0';
        document.body.style.transition = 'opacity 0.4s ease-in-out';
        
        // Trigger fade-in after a tiny delay
        requestAnimationFrame(function() {
            setTimeout(function() {
                document.body.style.opacity = '1';
            }, 10);
        });
    }

    // Initialize on page load
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', fadeIn);
    } else {
        fadeIn();
    }

    // Handle all internal link clicks
    document.addEventListener('click', function(e) {
        // Prevent multiple transitions
        if (isTransitioning) {
            e.preventDefault();
            return;
        }

        const link = e.target.closest('a');
        
        if (!link) return;
        
        const href = link.getAttribute('href');
        
        // Skip if:
        // - No href
        // - External link (starts with http:// or https://)
        // - Anchor link (starts with #)
        // - JavaScript link (starts with javascript:)
        // - Mailto/Tel links
        // - Download links
        // - Links with target="_blank"
        // - Button elements inside links
        if (!href || 
            href.startsWith('http://') || 
            href.startsWith('https://') || 
            href.startsWith('#') || 
            href.startsWith('javascript:') ||
            href.startsWith('mailto:') ||
            href.startsWith('tel:') ||
            link.hasAttribute('download') ||
            link.getAttribute('target') === '_blank' ||
            e.target.tagName === 'BUTTON') {
            return;
        }

        // Try to check if it's an internal link
        try {
            const url = new URL(href, window.location.origin);
            if (url.origin !== window.location.origin) {
                return;
            }
        } catch (e) {
            // If URL parsing fails, assume it's a relative path (internal link)
            // Continue with transition
        }

        // Prevent default navigation
        e.preventDefault();
        isTransitioning = true;

        // Add fade-out effect
        document.body.style.transition = 'opacity 0.35s ease-in-out';
        document.body.style.opacity = '0';

        // Navigate after fade-out
        setTimeout(function() {
            window.location.href = href;
        }, 350);
    });

    // Handle browser back/forward buttons
    window.addEventListener('pageshow', function(event) {
        isTransitioning = false;
        // If page was loaded from cache (back/forward button)
        if (event.persisted) {
            document.body.style.opacity = '0';
            requestAnimationFrame(function() {
                setTimeout(function() {
                    document.body.style.transition = 'opacity 0.4s ease-in-out';
                    document.body.style.opacity = '1';
                }, 10);
            });
        }
    });

    // Reset transition flag if page load is interrupted
    window.addEventListener('beforeunload', function() {
        isTransitioning = false;
    });
})();
