@extends('layouts.app')

@section('content')
<div class="container">
    <div class="row justify-content-center">
        <div class="col-lg-12">
            <!-- Header Card -->
            <div class="card shadow-sm mb-4 border-0">
                <div class="card-body p-4">
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <div>
                            <nav aria-label="breadcrumb">
                                <ol class="breadcrumb mb-2">
                                    <li class="breadcrumb-item"><a href="{{ route('quotation.index') }}" class="text-decoration-none">Quotations</a></li>
                                    <li class="breadcrumb-item active" aria-current="page">Create New</li>
                                </ol>
                            </nav>
                            <div class="d-flex align-items-center">
                                <div class="avatar-circle bg-primary text-white d-flex align-items-center justify-content-center me-3" style="width: 50px; height: 50px; font-size: 20px;">
                                    <i class="fas fa-file-invoice-dollar"></i>
                                </div>
                                <div>
                                    <h1 class="h3 fw-bold mb-0">Create New Quotation</h1>
                                    <p class="text-muted mb-0">Fill in the details to create a new quotation</p>
                                </div>
                            </div>
                        </div>
                        <div class="text-end">
                            <span class="badge bg-light text-dark fs-6 p-2">
                                <i class="fas fa-info-circle me-1"></i>
                                Required fields marked with *
                            </span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Form Card -->
            <div class="card shadow border-0">
                <div class="card-header bg-white py-3">
                    <h5 class="mb-0 fw-semibold">
                        <i class="fas fa-file-invoice text-primary me-2"></i>
                        Quotation Information
                    </h5>
                </div>
                
                <div class="card-body p-4">
                    @if ($errors->any())
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <div class="d-flex">
                                <i class="fas fa-exclamation-triangle me-3 mt-1"></i>
                                <div>
                                    <h6 class="alert-heading mb-2">Please fix the following errors:</h6>
                                    <ul class="mb-0 ps-3">
                                        @foreach ($errors->all() as $error)
                                            <li>{{ $error }}</li>
                                        @endforeach
                                    </ul>
                                </div>
                            </div>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    @endif

                    <form method="POST" action="{{ route('quotation.store') }}" id="createQuotationForm">
                        @csrf

                        <div class="row mb-4">
                            <!-- Customer Information -->
                            <div class="col-md-6">
                                <div class="form-section">
                                    <h6 class="section-title">
                                        <i class="fas fa-user me-2"></i>Customer Information
                                    </h6>
                                    
                                    <div class="mb-3">
                                        <label for="customer_id" class="form-label fw-semibold">
                                            Customer <span class="text-danger">*</span>
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-users text-primary"></i>
                                            </span>
                                            <select name="customer_id" 
                                                    id="customer_id"
                                                    class="form-select @error('customer_id') is-invalid @enderror" 
                                                    required>
                                                <option value="">Select Customer</option>
                                                @foreach($customers as $customer)
                                                    <option value="{{ $customer->id }}" 
                                                        {{ old('customer_id') == $customer->id ? 'selected' : '' }}>
                                                        {{ $customer->name }} 
                                                        @if($customer->phone)
                                                            ({{ $customer->phone }})
                                                        @endif
                                                    </option>
                                                @endforeach
                                            </select>
                                            @error('customer_id')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        <div class="mt-2">
                                            <a href="{{ route('customers.create') }}" class="text-decoration-none" target="_blank">
                                                <i class="fas fa-plus-circle me-1"></i>Add New Customer
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Quotation Details -->
                            <div class="col-md-6">
                                <div class="form-section">
                                    <h6 class="section-title">
                                        <i class="fas fa-calendar-alt me-2"></i>Quotation Details
                                    </h6>
                                    
                                    <div class="mb-3">
                                        <label for="quotation_no" class="form-label fw-semibold">
                                            Quotation Number
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-hashtag text-primary"></i>
                                            </span>
                                            <input type="text" 
                                                   name="quotation_no" 
                                                   id="quotation_no"
                                                   value="{{ old('quotation_no', \App\Models\Quotation::generateQuotationNo()) }}"
                                                   class="form-control @error('quotation_no') is-invalid @enderror" 
                                                   readonly>
                                            @error('quotation_no')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                    </div>

                                    <div class="mb-3">
                                        <label for="quotation_date" class="form-label fw-semibold">
                                            Quotation Date <span class="text-danger">*</span>
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-calendar text-primary"></i>
                                            </span>
                                            <input type="date" 
                                                   name="quotation_date" 
                                                   id="quotation_date"
                                                   value="{{ old('quotation_date', date('Y-m-d')) }}"
                                                   class="form-control @error('quotation_date') is-invalid @enderror" 
                                                   required>
                                            @error('quotation_date')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                    </div>

                                    <div class="mb-3">
                                        <label for="valid_till" class="form-label fw-semibold">
                                            Valid Till <span class="text-danger">*</span>
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-calendar-check text-primary"></i>
                                            </span>
                                            <input type="date" 
                                                   name="valid_till" 
                                                   id="valid_till"
                                                   value="{{ old('valid_till', date('Y-m-d', strtotime('+30 days'))) }}"
                                                   class="form-control @error('valid_till') is-invalid @enderror" 
                                                   required>
                                            @error('valid_till')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        <small class="text-muted">Quotation validity period</small>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Products Section -->
                        <div class="row mb-4">
                            <div class="col-12">
                                <div class="form-section">
                                    <h6 class="section-title">
                                        <i class="fas fa-boxes me-2"></i>Products
                                    </h6>
                                    
                                    <div class="table-responsive">
                                        <table class="table table-bordered" id="products_table">
                                            <thead class="table-light">
                                                <tr>
                                                    <th width="40%">Product <span class="text-danger">*</span></th>
                                                    <th width="15%">Quantity <span class="text-danger">*</span></th>
                                                    <th width="15%">Price (₹) <span class="text-danger">*</span></th>
                                                    <th width="15%">Subtotal (₹)</th>
                                                    <th width="15%">Actions</th>
                                                </tr>
                                            </thead>
                                            <tbody id="products_tbody">
                                                <!-- First row -->
                                                <tr id="product_row_1">
                                                    <td>
                                                        <select class="form-select product-select" 
                                                                name="products[0][product_id]" 
                                                                data-row="1" required>
                                                            <option value="">Select Product</option>
                                                            @foreach($products as $product)
                                                                <option value="{{ $product->id }}" 
                                                                        data-price="{{ $product->price }}" 
                                                                        data-stock="{{ $product->quantity }}">
                                                                    {{ $product->name }}
                                                                    @if($product->sku)
                                                                        ({{ $product->sku }})
                                                                    @endif
                                                                </option>
                                                            @endforeach
                                                        </select>
                                                    </td>
                                                    <td>
                                                        <input type="number" 
                                                               class="form-control quantity" 
                                                               name="products[0][quantity]" 
                                                               id="quantity_1" 
                                                               min="1" 
                                                               value="1" 
                                                               required>
                                                    </td>
                                                    <td>
                                                        <input type="number" 
                                                               class="form-control price" 
                                                               name="products[0][price]" 
                                                               id="price_1" 
                                                               step="0.01" 
                                                               min="0" 
                                                               value="0" 
                                                               required>
                                                    </td>
                                                    <td>
                                                        <input type="number" 
                                                               class="form-control subtotal" 
                                                               name="products[0][subtotal]" 
                                                               id="subtotal_1" 
                                                               step="0.01" 
                                                               min="0" 
                                                               value="0" 
                                                               readonly>
                                                    </td>
                                                    <td class="text-center">
                                                        <button type="button" class="btn btn-danger btn-sm" onclick="removeProductRow(1)" disabled>
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </td>
                                                </tr>
                                            </tbody>
                                            <tfoot>
                                                <tr>
                                                    <td colspan="5">
                                                        <button type="button" class="btn btn-primary btn-sm" onclick="addProductRow()">
                                                            <i class="fas fa-plus"></i> Add Product
                                                        </button>
                                                    </td>
                                                </tr>
                                            </tfoot>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Summary Section -->
                        <div class="row mb-4">
                            <div class="col-md-8">
                                <div class="form-section">
                                    <h6 class="section-title">
                                        <i class="fas fa-file-alt me-2"></i>Additional Information
                                    </h6>
                                    
                                    <div class="row">
                                        <div class="col-12 mb-3">
                                            <label for="notes" class="form-label fw-semibold">
                                                Notes / Terms & Conditions
                                            </label>
                                            <textarea name="notes" 
                                                    id="notes" 
                                                    class="form-control @error('notes') is-invalid @enderror" 
                                                    rows="4"
                                                    placeholder="Enter any additional notes, terms, or conditions...">{{ old('notes') }}</textarea>
                                            @error('notes')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-4">
                                <div class="form-section">
                                    <h6 class="section-title">
                                        <i class="fas fa-calculator me-2"></i>Amount Summary
                                    </h6>
                                    
                                    <div class="mb-3">
                                        <div class="d-flex justify-content-between mb-2">
                                            <span class="text-muted">Sub Amount:</span>
                                            <span class="fw-semibold" id="sub_amount_display">₹0.00</span>
                                            <input type="hidden" name="sub_amount" id="sub_amount" value="0">
                                        </div>
                                        
                                        <div class="d-flex justify-content-between mb-2">
                                            <span class="text-muted">Discount:</span>
                                            <div class="input-group input-group-sm" style="width: 120px;">
                                                <input type="number" 
                                                       name="discount" 
                                                       id="discount"
                                                       value="{{ old('discount', 0) }}"
                                                       class="form-control form-control-sm @error('discount') is-invalid @enderror" 
                                                       step="0.01"
                                                       min="0">
                                                <span class="input-group-text">₹</span>
                                            </div>
                                        </div>
                                        
                                        <div class="d-flex justify-content-between mb-2">
                                            <span class="text-muted">GST (18%):</span>
                                            <span class="fw-semibold text-warning" id="gst_amount_display">₹0.00</span>
                                            <input type="hidden" name="gst_amount" id="gst_amount" value="0">
                                        </div>
                                        
                                        <hr>
                                        
                                        <div class="d-flex justify-content-between mb-2">
                                            <span class="fw-bold">Grand Total:</span>
                                            <span class="fw-bold text-success fs-5" id="grand_total_display">₹0.00</span>
                                            <input type="hidden" name="grand_total" id="grand_total" value="0">
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Form Actions -->
                        <div class="row">
                            <div class="col-12">
                                <div class="d-flex justify-content-between align-items-center pt-4 border-top">
                                    <div>
                                        <a href="{{ route('quotation.index') }}" 
                                           class="btn btn-outline-secondary">
                                            <i class="fas fa-arrow-left me-1"></i>Back to Quotations
                                        </a>
                                    </div>
                                    <div>
                                        <button type="button" class="btn btn-outline-danger me-2" onclick="resetForm()">
                                            <i class="fas fa-redo me-1"></i>Reset Form
                                        </button>
                                        <button type="submit" class="btn btn-success px-4">
                                            <i class="fas fa-save me-1"></i>Create Quotation
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>

                <div class="card-footer bg-light py-3">
                    <div class="row align-items-center">
                        <div class="col-md-12 text-center">
                            <small class="text-muted">
                                <i class="fas fa-shield-alt me-1"></i>
                                Quotation information will be stored securely in the database.
                            </small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- JavaScript -->
<script>
    let rowCount = 1;
    
    // Initialize when page loads
    document.addEventListener('DOMContentLoaded', function() {
        // Add event listeners to first row
        setupRowEventListeners(1);
        
        // Add event listener for discount
        document.getElementById('discount').addEventListener('input', calculateTotals);
        
        // Calculate initial totals
        calculateTotals();
        
        // Set default valid till date (30 days from now)
        const today = new Date();
        const validTill = new Date();
        validTill.setDate(today.getDate() + 30);
        document.getElementById('valid_till').value = validTill.toISOString().split('T')[0];
    });
    
    // Add new product row
    function addProductRow() {
        rowCount++;
        const tbody = document.getElementById('products_tbody');
        const newRow = document.createElement('tr');
        newRow.id = `product_row_${rowCount}`;
        
        newRow.innerHTML = `
            <td>
                <select class="form-select product-select" 
                        name="products[${rowCount-1}][product_id]" 
                        data-row="${rowCount}" required>
                    <option value="">Select Product</option>
                    @foreach($products as $product)
                        <option value="{{ $product->id }}" 
                                data-price="{{ $product->price }}" 
                                data-stock="{{ $product->quantity }}">
                            {{ $product->name }}
                            @if($product->sku)
                                ({{ $product->sku }})
                            @endif
                        </option>
                    @endforeach
                </select>
            </td>
            <td>
                <input type="number" 
                       class="form-control quantity" 
                       name="products[${rowCount-1}][quantity]" 
                       id="quantity_${rowCount}" 
                       min="1" 
                       value="1" 
                       required>
            </td>
            <td>
                <input type="number" 
                       class="form-control price" 
                       name="products[${rowCount-1}][price]" 
                       id="price_${rowCount}" 
                       step="0.01" 
                       min="0" 
                       value="0" 
                       required>
            </td>
            <td>
                <input type="number" 
                       class="form-control subtotal" 
                       name="products[${rowCount-1}][subtotal]" 
                       id="subtotal_${rowCount}" 
                       step="0.01" 
                       min="0" 
                       value="0" 
                       readonly>
            </td>
            <td class="text-center">
                <button type="button" class="btn btn-danger btn-sm" onclick="removeProductRow(${rowCount})">
                    <i class="fas fa-trash"></i>
                </button>
            </td>
        `;
        
        tbody.appendChild(newRow);
        
        // Enable delete button for first row if this is the second row
        if (rowCount === 2) {
            document.querySelector('#product_row_1 .btn-danger').disabled = false;
        }
        
        // Setup event listeners for the new row
        setupRowEventListeners(rowCount);
    }
    
    // Setup event listeners for a row
    function setupRowEventListeners(rowId) {
        const productSelect = document.querySelector(`#product_row_${rowId} .product-select`);
        const quantityInput = document.getElementById(`quantity_${rowId}`);
        const priceInput = document.getElementById(`price_${rowId}`);
        
        productSelect.addEventListener('change', function() {
            fetchProductPrice(rowId);
        });
        
        quantityInput.addEventListener('input', function() {
            calculateRowTotal(rowId);
            checkStockAvailability(rowId);
        });
        
        priceInput.addEventListener('input', function() {
            calculateRowTotal(rowId);
        });
    }
    
    // Check stock availability
    function checkStockAvailability(rowId) {
        const selectElement = document.querySelector(`#product_row_${rowId} .product-select`);
        const quantityInput = document.getElementById(`quantity_${rowId}`);
        
        if (selectElement.value) {
            const selectedOption = selectElement.options[selectElement.selectedIndex];
            const stock = parseInt(selectedOption.getAttribute('data-stock')) || 0;
            const quantity = parseInt(quantityInput.value) || 0;
            
            if (quantity > stock) {
                quantityInput.classList.add('is-invalid');
                showStockWarning(rowId, stock);
            } else {
                quantityInput.classList.remove('is-invalid');
                removeStockWarning(rowId);
            }
        }
    }
    
    // Show stock warning
    function showStockWarning(rowId, stock) {
        let warningElement = document.querySelector(`#product_row_${rowId} .stock-warning`);
        if (!warningElement) {
            warningElement = document.createElement('div');
            warningElement.className = 'stock-warning text-danger small mt-1';
            document.querySelector(`#quantity_${rowId}`).parentNode.appendChild(warningElement);
        }
        warningElement.innerHTML = `<i class="fas fa-exclamation-triangle me-1"></i> Only ${stock} units available in stock`;
    }
    
    // Remove stock warning
    function removeStockWarning(rowId) {
        const warningElement = document.querySelector(`#product_row_${rowId} .stock-warning`);
        if (warningElement) {
            warningElement.remove();
        }
    }
    
    // Remove product row
    function removeProductRow(rowId) {
        const row = document.getElementById(`product_row_${rowId}`);
        if (row && rowCount > 1) {
            row.remove();
            rowCount--;
            
            // If we're back to only one row, disable its delete button
            if (rowCount === 1) {
                document.querySelector('#product_row_1 .btn-danger').disabled = true;
            }
            
            calculateTotals();
        }
    }
    
    // Fetch product price
    function fetchProductPrice(rowId) {
        const selectElement = document.querySelector(`#product_row_${rowId} .product-select`);
        const selectedOption = selectElement.options[selectElement.selectedIndex];
        const priceInput = document.getElementById(`price_${rowId}`);
        
        if (selectedOption.value) {
            const price = selectedOption.getAttribute('data-price');
            priceInput.value = price;
            
            // Check stock availability
            checkStockAvailability(rowId);
            
            // Calculate row total
            calculateRowTotal(rowId);
        } else {
            priceInput.value = '';
            document.getElementById(`subtotal_${rowId}`).value = '';
            calculateTotals();
        }
    }
    
    // Calculate row total
    function calculateRowTotal(rowId) {
        const priceInput = document.getElementById(`price_${rowId}`);
        const quantityInput = document.getElementById(`quantity_${rowId}`);
        const subtotalInput = document.getElementById(`subtotal_${rowId}`);
        
        const price = parseFloat(priceInput.value) || 0;
        const quantity = parseInt(quantityInput.value) || 0;
        
        const subtotal = price * quantity;
        subtotalInput.value = subtotal.toFixed(2);
        
        calculateTotals();
    }
    
    // Calculate all totals
    function calculateTotals() {
        let subTotal = 0;
        const subtotalInputs = document.querySelectorAll('.subtotal');
        
        subtotalInputs.forEach(input => {
            const value = parseFloat(input.value) || 0;
            subTotal += value;
        });
        
        const discount = parseFloat(document.getElementById('discount').value) || 0;
        const gstRate = 0.18; // 18% GST
        const gstAmount = (subTotal - discount) * gstRate;
        const grandTotal = (subTotal - discount) + gstAmount;
        
        // Update display fields
        document.getElementById('sub_amount_display').textContent = '₹' + subTotal.toFixed(2);
        document.getElementById('sub_amount').value = subTotal.toFixed(2);
        
        document.getElementById('gst_amount_display').textContent = '₹' + gstAmount.toFixed(2);
        document.getElementById('gst_amount').value = gstAmount.toFixed(2);
        
        document.getElementById('grand_total_display').textContent = '₹' + grandTotal.toFixed(2);
        document.getElementById('grand_total').value = grandTotal.toFixed(2);
    }
    
    // Reset form
    function resetForm() {
        if (confirm('Are you sure you want to reset the form? All data will be lost.')) {
            document.getElementById('createQuotationForm').reset();
            
            // Reset product rows to just one
            const tbody = document.getElementById('products_tbody');
            while (tbody.children.length > 1) {
                tbody.removeChild(tbody.lastChild);
            }
            
            // Reset the first row
            const firstRowSelect = document.querySelector('#product_row_1 .product-select');
            const firstRowPrice = document.getElementById('price_1');
            const firstRowQuantity = document.getElementById('quantity_1');
            const firstRowSubtotal = document.getElementById('subtotal_1');
            
            firstRowSelect.value = '';
            firstRowPrice.value = '';
            firstRowQuantity.value = '1';
            firstRowSubtotal.value = '';
            
            // Disable delete button for first row
            document.querySelector('#product_row_1 .btn-danger').disabled = true;
            
            // Reset valid till date
            const today = new Date();
            const validTill = new Date();
            validTill.setDate(today.getDate() + 30);
            document.getElementById('valid_till').value = validTill.toISOString().split('T')[0];
            
            rowCount = 1;
            calculateTotals();
        }
    }
    
    // Form validation
    document.getElementById('createQuotationForm').addEventListener('submit', function(e) {
        let valid = true;
        let errorMessages = [];
        
        // Check if at least one product is selected
        const productSelects = this.querySelectorAll('.product-select');
        let hasProduct = false;
        
        productSelects.forEach(select => {
            if (select.value) hasProduct = true;
        });
        
        if (!hasProduct) {
            valid = false;
            errorMessages.push('Please add at least one product to the quotation.');
        }
        
        // Validate customer
        const customerId = document.getElementById('customer_id').value;
        if (!customerId) {
            valid = false;
            errorMessages.push('Please select a customer.');
        }
        
        // Validate dates
        const quotationDate = document.getElementById('quotation_date').value;
        const validTill = document.getElementById('valid_till').value;
        
        if (!quotationDate) {
            valid = false;
            errorMessages.push('Please select quotation date.');
        }
        
        if (!validTill) {
            valid = false;
            errorMessages.push('Please select valid till date.');
        }
        
        if (quotationDate && validTill) {
            const qDate = new Date(quotationDate);
            const vDate = new Date(validTill);
            
            if (vDate < qDate) {
                valid = false;
                errorMessages.push('Valid till date cannot be before quotation date.');
            }
        }
        
        // Validate all product rows have valid data
        const quantityInputs = this.querySelectorAll('.quantity');
        quantityInputs.forEach(input => {
            if (parseInt(input.value) <= 0) {
                valid = false;
                errorMessages.push('Please enter valid quantity for all products.');
            }
        });
        
        const priceInputs = this.querySelectorAll('.price');
        priceInputs.forEach(input => {
            if (parseFloat(input.value) <= 0) {
                valid = false;
                errorMessages.push('Please enter valid price for all products.');
            }
        });
        
        // Check stock availability
        const stockWarnings = document.querySelectorAll('.stock-warning');
        if (stockWarnings.length > 0) {
            valid = false;
            errorMessages.push('Some products have insufficient stock. Please adjust quantities.');
        }
        
        if (!valid) {
            e.preventDefault();
            alert(errorMessages.join('\n'));
        }
    });
</script>

<style>
    .avatar-circle {
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: bold;
    }

    .form-section {
        background: #f8f9fa;
        border-radius: 10px;
        padding: 1.5rem;
        margin-bottom: 1.5rem;
        border: 1px solid #e9ecef;
    }

    .section-title {
        color: #495057;
        font-size: 1rem;
        font-weight: 600;
        margin-bottom: 1.5rem;
        padding-bottom: 0.5rem;
        border-bottom: 2px solid #dee2e6;
    }

    .form-label {
        font-weight: 500;
        margin-bottom: 0.5rem;
        color: #495057;
    }

    .input-group-text {
        border-right: none;
        background-color: #f8f9fa !important;
    }

    .form-control, .form-select {
        border-left: none;
        padding-left: 0;
    }

    .form-control:focus, .form-select:focus {
        box-shadow: none;
        border-color: #86b7fe;
    }

    .card {
        border: none;
        border-radius: 15px;
        overflow: hidden;
    }

    .card-header {
        background-color: white;
        border-bottom: 1px solid rgba(0,0,0,.125);
        font-weight: 600;
    }

    .alert {
        border: none;
        border-radius: 10px;
    }

    .breadcrumb-item a {
        color: #6c757d;
        transition: color 0.2s;
    }

    .breadcrumb-item a:hover {
        color: #0d6efd;
        text-decoration: underline;
    }

    .btn-success {
        background: linear-gradient(135deg, #198754 0%, #157347 100%);
        border: none;
        padding: 0.5rem 2rem;
        font-weight: 500;
    }

    .btn-success:hover {
        background: linear-gradient(135deg, #157347 0%, #0d5c3a 100%);
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(25, 135, 84, 0.2);
    }

    .badge {
        font-weight: 500;
        padding: 0.5rem 1rem;
        border-radius: 20px;
    }

    .card-footer {
        background-color: #f8f9fa;
        border-top: 1px solid #e9ecef;
    }

    .table th {
        background-color: #f8f9fa;
        font-weight: 600;
    }
    
    .table td {
        vertical-align: middle;
    }
    
    input[type="number"] {
        text-align: right;
    }
    
    .subtotal {
        font-weight: bold;
        color: #198754;
    }
    
    /* Style for Add Product button */
    .btn-primary {
        transition: all 0.3s ease;
    }
    
    .btn-primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
    }
    
    #grand_total_display {
        background-color: #f0f9ff;
        font-size: 1.1rem;
        border-radius: 4px;
        padding: 0.375rem 0.75rem;
    }
    
    .stock-warning {
        font-size: 0.8rem;
    }
    
    /* Status badge styles */
    .status-badge {
        padding: 0.25rem 0.75rem;
        border-radius: 20px;
        font-size: 0.875rem;
        font-weight: 500;
    }
    
    .status-draft {
        background-color: #e9ecef;
        color: #495057;
    }
    
    .status-sent {
        background-color: #cfe2ff;
        color: #084298;
    }
    
    .status-accepted {
        background-color: #d1e7dd;
        color: #0a3622;
    }
    
    .status-rejected {
        background-color: #f8d7da;
        color: #58151c;
    }
    
    .status-expired {
        background-color: #e2e3e5;
        color: #2b2f32;
    }
    
    /* Notes textarea */
    #notes {
        resize: vertical;
        min-height: 100px;
    }
</style>
@endsection