@extends('layouts.app')

@section('content')
<div class="container">
    <div class="row justify-content-center">
        <div class="col-lg-10">
            <!-- Header Card -->
            <div class="card shadow-sm mb-4 border-0">
                <div class="card-body p-4">
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <div>
                            <nav aria-label="breadcrumb">
                                <ol class="breadcrumb mb-2">
                                    <li class="breadcrumb-item"><a href="{{ route('products.index') }}" class="text-decoration-none">Products</a></li>
                                    <li class="breadcrumb-item"><a href="{{ route('products.show', $product->id) }}" class="text-decoration-none">{{ $product->name }}</a></li>
                                    <li class="breadcrumb-item active" aria-current="page">Edit</li>
                                </ol>
                            </nav>
                            <div class="d-flex align-items-center">
                                <div class="avatar-circle bg-primary text-white d-flex align-items-center justify-content-center me-3" style="width: 50px; height: 50px; font-size: 20px;">
                                    @if($product->image)
                                        <img src="{{ asset('storage/'.$product->image) }}" 
                                             alt="{{ $product->name }}" 
                                             style="width: 100%; height: 100%; object-fit: cover; border-radius: 50%;">
                                    @else
                                        <i class="fas fa-box-open"></i>
                                    @endif
                                </div>
                                <div>
                                    <h1 class="h3 fw-bold mb-0">Edit Product</h1>
                                    <p class="text-muted mb-0">ID: {{ str_pad($product->id, 6, '0', STR_PAD_LEFT) }} | SKU: {{ $product->sku ?: 'Not set' }}</p>
                                </div>
                            </div>
                        </div>
                        <div class="text-end">
                            <span class="badge bg-light text-dark fs-6 p-2">
                                <i class="fas fa-calendar-alt me-1"></i>
                                Created: {{ $product->created_at->format('M d, Y') }}
                            </span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Form Card -->
            <div class="card shadow border-0">
                <div class="card-header bg-white py-3">
                    <h5 class="mb-0 fw-semibold">
                        <i class="fas fa-edit text-primary me-2"></i>
                        Update Product Information
                    </h5>
                </div>
                
                <div class="card-body p-4">
                    @if ($errors->any())
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <div class="d-flex">
                                <i class="fas fa-exclamation-triangle me-3 mt-1"></i>
                                <div>
                                    <h6 class="alert-heading mb-2">Please fix the following errors:</h6>
                                    <ul class="mb-0 ps-3">
                                        @foreach ($errors->all() as $error)
                                            <li>{{ $error }}</li>
                                        @endforeach
                                    </ul>
                                </div>
                            </div>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    @endif

                    @if(session('success'))
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <i class="fas fa-check-circle me-2"></i>
                            {{ session('success') }}
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    @endif

                    <form method="POST" action="{{ route('products.update', $product->id) }}" enctype="multipart/form-data" id="editProductForm">
                        @csrf
                        @method('PUT')

                        <div class="row">
                            <!-- Basic Information Section -->
                            <div class="col-md-6 mb-4">
                                <div class="form-section">
                                    <h6 class="section-title">
                                        <i class="fas fa-info-circle me-2"></i>Basic Information
                                    </h6>
                                    
                                    <div class="mb-3">
                                        <label for="name" class="form-label fw-semibold">
                                            Product Name <span class="text-danger">*</span>
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-tag text-primary"></i>
                                            </span>
                                            <input type="text" 
                                                   name="name" 
                                                   id="name"
                                                   value="{{ old('name', $product->name) }}"
                                                   class="form-control @error('name') is-invalid @enderror" 
                                                   required
                                                   placeholder="Enter product name">
                                            @error('name')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                    </div>

                                    <div class="mb-3">
                                        <label for="sku" class="form-label fw-semibold">
                                            SKU (Stock Keeping Unit)
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-barcode text-primary"></i>
                                            </span>
                                            <input type="text" 
                                                   name="sku" 
                                                   id="sku"
                                                   value="{{ old('sku', $product->sku) }}"
                                                   class="form-control @error('sku') is-invalid @enderror" 
                                                   placeholder="e.g., PROD-001">
                                            @error('sku')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        <small class="text-muted">Unique identifier for the product</small>
                                    </div>

                                    <!-- CATEGORY DROPDOWN - UPDATED -->
                                    <div class="mb-3">
                                        <label for="category_id" class="form-label fw-semibold">
                                            Category <span class="text-danger">*</span>
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-folder text-primary"></i>
                                            </span>
                                            <select name="category_id" 
                                                    id="category_id"
                                                    class="form-select @error('category_id') is-invalid @enderror" 
                                                    required>
                                                <option value="">Select Category</option>
                                                @foreach($categories as $category)
                                                    <option value="{{ $category->id }}" 
                                                        {{ old('category_id', $product->category_id) == $category->id ? 'selected' : '' }}>
                                                        {{ $category->name }}
                                                        @if($category->description)
                                                            - {{ Str::limit($category->description, 30) }}
                                                        @endif
                                                    </option>
                                                @endforeach
                                            </select>
                                            @error('category_id')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        <div class="mt-2">
                                            <a href="{{ route('categories.create') }}" class="text-decoration-none" target="_blank">
                                                <i class="fas fa-plus-circle me-1"></i>Add New Category
                                            </a>
                                        </div>
                                    </div>
                                    <!-- END CATEGORY DROPDOWN -->
                                </div>
                            </div>

                            <!-- Pricing & Inventory Section -->
                            <div class="col-md-6 mb-4">
                                <div class="form-section">
                                    <h6 class="section-title">
                                        <i class="fas fa-chart-line me-2"></i>Pricing & Inventory
                                    </h6>
                                    
                                    <div class="mb-3">
                                        <label for="price" class="form-label fw-semibold">
                                            Price <span class="text-danger">*</span>
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-dollar-sign text-primary"></i>
                                            </span>
                                            <input type="number" 
                                                   name="price" 
                                                   id="price"
                                                   value="{{ old('price', $product->price) }}"
                                                   class="form-control @error('price') is-invalid @enderror" 
                                                   step="0.01"
                                                   min="0"
                                                   required
                                                   placeholder="0.00">
                                            @error('price')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        <small class="text-muted">Current value: ${{ number_format($product->price, 2) }}</small>
                                    </div>

                                    <div class="mb-3">
                                        <label for="quantity" class="form-label fw-semibold">
                                            Quantity <span class="text-danger">*</span>
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-boxes text-primary"></i>
                                            </span>
                                            <input type="number" 
                                                   name="quantity" 
                                                   id="quantity"
                                                   value="{{ old('quantity', $product->quantity) }}"
                                                   class="form-control @error('quantity') is-invalid @enderror" 
                                                   min="0"
                                                   required
                                                   placeholder="0">
                                            @error('quantity')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        <small class="text-muted">Current stock: {{ $product->quantity }} {{ $product->unit ?: 'units' }}</small>
                                    </div>

                                    <div class="mb-3">
                                        <label for="unit" class="form-label fw-semibold">
                                            Unit
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-weight text-primary"></i>
                                            </span>
                                            <input type="text" 
                                                   name="unit" 
                                                   id="unit"
                                                   value="{{ old('unit', $product->unit) }}"
                                                   class="form-control @error('unit') is-invalid @enderror" 
                                                   placeholder="e.g., pcs, kg, box">
                                            @error('unit')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        <small class="text-muted">Unit of measurement for the product</small>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Description Section -->
                        <div class="row mb-4">
                            <div class="col-12">
                                <div class="form-section">
                                    <h6 class="section-title">
                                        <i class="fas fa-align-left me-2"></i>Description
                                    </h6>
                                    <div class="mb-3">
                                        <label for="description" class="form-label fw-semibold">
                                            Product Description
                                        </label>
                                        <textarea name="description" 
                                                  id="description"
                                                  class="form-control @error('description') is-invalid @enderror" 
                                                  rows="4"
                                                  placeholder="Enter detailed product description">{{ old('description', $product->description) }}</textarea>
                                        @error('description')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Status & Image Section -->
                        <div class="row">
                            <div class="col-md-6 mb-4">
                                <div class="form-section">
                                    <h6 class="section-title">
                                        <i class="fas fa-toggle-on me-2"></i>Status & Settings
                                    </h6>
                                    
                                    <div class="mb-3">
                                        <label for="status" class="form-label fw-semibold">
                                            Product Status
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-power-off text-primary"></i>
                                            </span>
                                            <select name="status" 
                                                    id="status"
                                                    class="form-control @error('status') is-invalid @enderror">
                                                <option value="active" {{ old('status', $product->status) == 'active' ? 'selected' : '' }}>Active</option>
                                                <option value="inactive" {{ old('status', $product->status) == 'inactive' ? 'selected' : '' }}>Inactive</option>
                                                <option value="out_of_stock" {{ old('status', $product->status) == 'out_of_stock' ? 'selected' : '' }}>Out of Stock</option>
                                            </select>
                                            @error('status')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        <small class="text-muted">Current status: 
                                            <span class="badge bg-{{ $product->status == 'active' ? 'success' : ($product->status == 'out_of_stock' ? 'warning' : 'secondary') }}">
                                                {{ ucfirst(str_replace('_', ' ', $product->status)) }}
                                            </span>
                                        </small>
                                    </div>

                                    <div class="mb-3">
                                        <label class="form-label fw-semibold">
                                            Product Statistics
                                        </label>
                                        <div class="row g-2">
                                            <div class="col-6">
                                                <div class="p-3 bg-light rounded">
                                                    <small class="text-muted d-block">Last Updated</small>
                                                    <strong>{{ $product->updated_at->diffForHumans() }}</strong>
                                                </div>
                                            </div>
                                            <div class="col-6">
                                                <div class="p-3 bg-light rounded">
                                                    <small class="text-muted d-block">Total Value</small>
                                                    <strong id="totalValue">${{ number_format($product->price * $product->quantity, 2) }}</strong>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-6 mb-4">
                                <div class="form-section">
                                    <h6 class="section-title">
                                        <i class="fas fa-image me-2"></i>Product Image
                                    </h6>
                                    
                                    <div class="mb-3">
                                        <label for="image" class="form-label fw-semibold">
                                            Update Image
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-camera text-primary"></i>
                                            </span>
                                            <input type="file" 
                                                   name="image" 
                                                   id="image"
                                                   class="form-control @error('image') is-invalid @enderror"
                                                   accept="image/*">
                                            @error('image')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        <small class="text-muted">Max file size: 2MB. Allowed: JPG, PNG, GIF</small>
                                        
                                        <!-- Current Image -->
                                        @if($product->image)
                                            <div class="mt-3">
                                                <p class="mb-2"><small>Current Image:</small></p>
                                                <div class="d-flex align-items-start gap-3">
                                                    <img src="{{ asset('storage/'.$product->image) }}" 
                                                         class="img-thumbnail" 
                                                         style="max-height: 150px;"
                                                         alt="{{ $product->name }}">
                                                    <div>
                                                        <small class="text-muted d-block">File: {{ basename($product->image) }}</small>
                                                        <div class="form-check mt-2">
                                                            <input class="form-check-input" type="checkbox" name="remove_image" id="remove_image" value="1">
                                                            <label class="form-check-label text-danger" for="remove_image">
                                                                <small>Remove current image</small>
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        @else
                                            <div class="mt-3">
                                                <p class="text-muted mb-2">
                                                    <i class="fas fa-image text-secondary"></i> No image uploaded
                                                </p>
                                            </div>
                                        @endif
                                        
                                        <!-- New Image Preview -->
                                        <div class="mt-3" id="newImagePreviewContainer" style="display: none;">
                                            <p class="mb-2"><small>New Image Preview:</small></p>
                                            <img id="newImagePreview" class="img-thumbnail" style="max-height: 150px;">
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Form Actions -->
                        <div class="row">
                            <div class="col-12">
                                <div class="d-flex justify-content-between align-items-center pt-4 border-top">
                                    <div>
                                        <a href="{{ route('products.show', $product->id) }}" 
                                           class="btn btn-outline-secondary me-2">
                                            <i class="fas fa-eye me-1"></i>View Product
                                        </a>
                                        <a href="{{ route('products.index') }}" 
                                           class="btn btn-outline-secondary">
                                            <i class="fas fa-arrow-left me-1"></i>Back to Products
                                        </a>
                                    </div>
                                    <div>
                                        <button type="reset" class="btn btn-outline-danger me-2">
                                            <i class="fas fa-redo me-1"></i>Reset Changes
                                        </button>
                                        <button type="submit" class="btn btn-success px-4">
                                            <i class="fas fa-save me-1"></i>Update Product
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>

                <!-- Card Footer -->
                <div class="card-footer bg-light py-3">
                    <div class="row align-items-center">
                        <div class="col-md-6">
                            <small class="text-muted">
                                <i class="fas fa-info-circle me-1"></i>
                                Last updated: {{ $product->updated_at->diffForHumans() }}
                            </small>
                        </div>
                        <div class="col-md-6 text-end">
                            <form action="{{ route('products.destroy', $product->id) }}" 
                                  method="POST" 
                                  class="d-inline"
                                  id="deleteForm">
                                @csrf
                                @method('DELETE')
                                <button type="button" 
                                        class="btn btn-sm btn-outline-danger" 
                                        id="deleteProductBtn"
                                        data-product-name="{{ $product->name }}">
                                    <i class="fas fa-trash me-1"></i>Delete Product
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
    // Form validation
    document.getElementById('editProductForm').addEventListener('submit', function(e) {
        const requiredFields = this.querySelectorAll('[required]');
        let valid = true;
        
        requiredFields.forEach(field => {
            if (!field.value.trim() && field.type !== 'file') {
                valid = false;
                if (!field.classList.contains('is-invalid')) {
                    field.classList.add('is-invalid');
                }
            }
        });
        
        // Validate price
        const priceField = document.getElementById('price');
        if (priceField.value < 0) {
            valid = false;
            if (!priceField.classList.contains('is-invalid')) {
                priceField.classList.add('is-invalid');
                priceField.nextElementSibling.textContent = 'Price cannot be negative';
            }
        }
        
        // Validate quantity
        const quantityField = document.getElementById('quantity');
        if (quantityField.value < 0) {
            valid = false;
            if (!quantityField.classList.contains('is-invalid')) {
                quantityField.classList.add('is-invalid');
                quantityField.nextElementSibling.textContent = 'Quantity cannot be negative';
            }
        }
        
        if (!valid) {
            e.preventDefault();
            const alertDiv = document.createElement('div');
            alertDiv.className = 'alert alert-danger alert-dismissible fade show mt-3';
            alertDiv.innerHTML = `
                <i class="fas fa-exclamation-triangle me-2"></i>
                Please fix the errors in the form before submitting.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            this.querySelector('.card-body').prepend(alertDiv);
            
            // Scroll to first error
            const firstError = this.querySelector('.is-invalid');
            if (firstError) {
                firstError.scrollIntoView({ behavior: 'smooth', block: 'center' });
            }
        }
    });

    // Remove validation styling on input
    document.querySelectorAll('input, textarea, select').forEach(field => {
        field.addEventListener('input', function() {
            if (this.classList.contains('is-invalid')) {
                this.classList.remove('is-invalid');
                const feedback = this.nextElementSibling;
                if (feedback && feedback.classList.contains('invalid-feedback')) {
                    feedback.textContent = '';
                }
            }
        });
    });

    // New image preview
    document.getElementById('image').addEventListener('change', function(e) {
        const file = e.target.files[0];
        const preview = document.getElementById('newImagePreview');
        const previewContainer = document.getElementById('newImagePreviewContainer');
        const removeImageCheckbox = document.getElementById('remove_image');
        
        if (file) {
            const reader = new FileReader();
            
            reader.onload = function(e) {
                preview.src = e.target.result;
                previewContainer.style.display = 'block';
                
                // Uncheck remove image if new image is selected
                if (removeImageCheckbox) {
                    removeImageCheckbox.checked = false;
                }
            }
            
            reader.readAsDataURL(file);
            
            // Validate file size (2MB)
            if (file.size > 2 * 1024 * 1024) {
                alert('File size exceeds 2MB limit. Please choose a smaller file.');
                this.value = '';
                previewContainer.style.display = 'none';
            }
            
            // Validate file type
            const validTypes = ['image/jpeg', 'image/png', 'image/gif'];
            if (!validTypes.includes(file.type)) {
                alert('Please select a valid image file (JPG, PNG, or GIF).');
                this.value = '';
                previewContainer.style.display = 'none';
            }
        } else {
            previewContainer.style.display = 'none';
        }
    });

    // Toggle new image preview when remove image is checked
    const removeImageCheckbox = document.getElementById('remove_image');
    if (removeImageCheckbox) {
        removeImageCheckbox.addEventListener('change', function() {
            if (this.checked) {
                // Clear file input
                document.getElementById('image').value = '';
                // Hide new image preview
                document.getElementById('newImagePreviewContainer').style.display = 'none';
            }
        });
    }

    // Auto-calculate total value
    const priceField = document.getElementById('price');
    const quantityField = document.getElementById('quantity');
    
    function calculateTotalValue() {
        const price = parseFloat(priceField.value) || 0;
        const quantity = parseFloat(quantityField.value) || 0;
        const totalValue = price * quantity;
        
        // Update total value display
        const totalValueElement = document.getElementById('totalValue');
        if (totalValueElement) {
            totalValueElement.textContent = `$${totalValue.toFixed(2)}`;
        }
    }
    
    priceField.addEventListener('input', calculateTotalValue);
    quantityField.addEventListener('input', calculateTotalValue);

    // Character counter for description
    const descriptionTextarea = document.getElementById('description');
    if (descriptionTextarea) {
        const counter = document.createElement('small');
        counter.className = 'text-muted float-end';
        counter.textContent = `${descriptionTextarea.value.length}/1000`;
        
        descriptionTextarea.parentNode.appendChild(counter);
        
        descriptionTextarea.addEventListener('input', function() {
            counter.textContent = `${this.value.length}/1000`;
            
            if (this.value.length > 1000) {
                counter.classList.remove('text-muted');
                counter.classList.add('text-danger');
            } else {
                counter.classList.remove('text-danger');
                counter.classList.add('text-muted');
            }
        });
    }

    // Delete confirmation
    document.getElementById('deleteProductBtn').addEventListener('click', function() {
        const productName = this.dataset.productName;
        
        if (confirm(`Are you sure you want to delete "${productName}"? This action cannot be undone!`)) {
            document.getElementById('deleteForm').submit();
        }
    });

    // Form reset confirmation
    document.querySelector('button[type="reset"]').addEventListener('click', function(e) {
        if (confirm('Are you sure you want to reset all changes? All unsaved changes will be lost.')) {
            // Clear new image preview
            document.getElementById('newImagePreviewContainer').style.display = 'none';
            // Uncheck remove image
            const removeImageCheckbox = document.getElementById('remove_image');
            if (removeImageCheckbox) {
                removeImageCheckbox.checked = false;
            }
            return true;
        } else {
            e.preventDefault();
            return false;
        }
    });

    // Category search enhancement
    document.getElementById('category_id').addEventListener('focus', function() {
        this.size = 5; // Show more options when focused
    });

    document.getElementById('category_id').addEventListener('blur', function() {
        this.size = 1; // Reset to single line
    });

    document.getElementById('category_id').addEventListener('keyup', function() {
        const searchTerm = this.value.toLowerCase();
        const options = this.options;
        
        // Simple search highlight (for better UX, consider Select2)
        for (let i = 0; i < options.length; i++) {
            const option = options[i];
            const text = option.text.toLowerCase();
            if (text.includes(searchTerm) && searchTerm.length > 0) {
                option.style.backgroundColor = '#e7f3ff';
            } else {
                option.style.backgroundColor = '';
            }
        }
    });
</script>
@endpush

@push('styles')
<style>
    .avatar-circle {
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: bold;
        overflow: hidden;
    }

    .form-section {
        background: #f8f9fa;
        border-radius: 10px;
        padding: 1.5rem;
        margin-bottom: 1.5rem;
        border: 1px solid #e9ecef;
    }

    .section-title {
        color: #495057;
        font-size: 1rem;
        font-weight: 600;
        margin-bottom: 1.5rem;
        padding-bottom: 0.5rem;
        border-bottom: 2px solid #dee2e6;
    }

    .form-label {
        font-weight: 500;
        margin-bottom: 0.5rem;
        color: #495057;
    }

    .input-group-text {
        border-right: none;
        background-color: #f8f9fa !important;
    }

    .form-control, .form-select {
        border-left: none;
        padding-left: 0;
    }

    .form-control:focus, .form-select:focus {
        box-shadow: none;
        border-color: #86b7fe;
    }

    .form-control:focus + .input-group-text,
    .form-select:focus + .input-group-text {
        border-color: #86b7fe;
    }

    .card {
        border: none;
        border-radius: 15px;
        overflow: hidden;
    }

    .card-header {
        background-color: white;
        border-bottom: 1px solid rgba(0,0,0,.125);
        font-weight: 600;
    }

    .alert {
        border: none;
        border-radius: 10px;
    }

    .alert-danger {
        background-color: #f8d7da;
        border-left: 4px solid #dc3545;
    }

    .alert-success {
        background-color: #d1e7dd;
        border-left: 4px solid #198754;
    }

    .breadcrumb {
        background: transparent;
        padding: 0;
        margin-bottom: 0.5rem;
    }

    .breadcrumb-item a {
        color: #6c757d;
        transition: color 0.2s;
    }

    .breadcrumb-item a:hover {
        color: #0d6efd;
        text-decoration: underline;
    }

    .btn-success {
        background: linear-gradient(135deg, #198754 0%, #157347 100%);
        border: none;
        padding: 0.5rem 2rem;
        font-weight: 500;
    }

    .btn-success:hover {
        background: linear-gradient(135deg, #157347 0%, #0d5c3a 100%);
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(25, 135, 84, 0.2);
    }

    .btn-outline-secondary:hover {
        transform: translateY(-1px);
    }

    .btn-outline-danger:hover {
        background-color: #dc3545;
        color: white;
    }

    .btn-sm {
        padding: 0.25rem 0.5rem;
        font-size: 0.875rem;
    }

    textarea {
        resize: vertical;
        min-height: 100px;
    }

    .badge {
        font-weight: 500;
        padding: 0.35rem 0.65rem;
        border-radius: 20px;
    }

    .bg-light.rounded {
        background-color: #f8f9fa !important;
        border: 1px solid #e9ecef;
    }

    .card-footer {
        background-color: #f8f9fa;
        border-top: 1px solid #e9ecef;
    }

    .img-thumbnail {
        border: 1px solid #dee2e6;
        border-radius: 8px;
        padding: 5px;
        background-color: white;
        object-fit: cover;
    }

    input[type="number"]::-webkit-inner-spin-button,
    input[type="number"]::-webkit-outer-spin-button {
        -webkit-appearance: none;
        margin: 0;
    }

    input[type="number"] {
        -moz-appearance: textfield;
    }

    .form-check-input:checked {
        background-color: #dc3545;
        border-color: #dc3545;
    }

    .bg-success {
        background-color: #198754 !important;
    }

    .bg-warning {
        background-color: #ffc107 !important;
    }

    .bg-secondary {
        background-color: #6c757d !important;
    }

    /* Category dropdown styling */
    .form-select option {
        padding: 8px 12px;
    }

    .form-select option:hover {
        background-color: #e7f3ff !important;
    }

    .form-select option:checked {
        background-color: #0d6efd !important;
        color: white !important;
    }
</style>
@endpush
@endsection