@extends('layouts.app')

@section('content')
<div class="container">
    <h1>Create Invoice</h1>
    
    <form id="invoiceForm" method="POST" action="{{ route('invoices.store') }}">
        @csrf
        
        <div class="row">
            <!-- Left Column - Invoice Details -->
            <div class="col-md-6">
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0">Invoice Details</h5>
                    </div>
                    <div class="card-body">
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="invoice_no" class="form-label">Invoice No</label>
                                <input type="text" class="form-control" id="invoice_no" name="invoice_no" 
                                       value="INV-{{ time() }}" readonly>
                            </div>
                            <div class="col-md-6">
                                <label for="invoice_date" class="form-label">Invoice Date</label>
                                <input type="date" class="form-control" id="invoice_date" name="invoice_date" 
                                       value="{{ date('Y-m-d') }}" required>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="customer_id" class="form-label">Customer</label>
                            <div class="input-group">
                                <select class="form-select" id="customer_id" name="customer_id" required>
                                    <option value="">--SELECT CUSTOMER--</option>
                                    @foreach($customers as $customer)
                                        <option value="{{ $customer->id }}">{{ $customer->name }}</option>
                                    @endforeach
                                </select>
                                <a href="{{ url('customers/create') }}" target="_blank" class="btn btn-outline-primary">
                                    <i class="fas fa-plus"></i> Add New
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Right Column - Payment Details -->
            <div class="col-md-6">
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0">Payment Details</h5>
                    </div>
                    <div class="card-body">
                        <div class="row mb-3">
                            <div class="col-md-4">
                                <label class="form-label">Sub Amount</label>
                                <div class="form-control" id="sub_amount_display">₹0.00</div>
                                <input type="hidden" id="sub_amount" name="sub_amount" value="0">
                            </div>
                            <div class="col-md-4">
                                <label for="discount" class="form-label">Discount (₹)</label>
                                <input type="number" class="form-control" id="discount" name="discount" 
                                       value="0" min="0" step="0.01">
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">Total Amount</label>
                                <div class="form-control" id="total_amount_display">₹0.00</div>
                                <input type="hidden" id="total_amount" name="total_amount" value="0">
                            </div>
                        </div>
                        
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label class="form-label">GST (18%)</label>
                                <div class="form-control" id="gst_amount_display">₹0.00</div>
                                <input type="hidden" id="gst_amount" name="gst_amount" value="0">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Grand Total</label>
                                <div class="form-control fw-bold" id="grand_total_display">₹0.00</div>
                                <input type="hidden" id="grand_total" name="grand_total" value="0">
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Products Section -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">Products</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table" id="products_table">
                        <thead>
                            <tr>
                                <th width="30%">Product</th>
                                <th width="20%">Price (₹)</th>
                                <th width="20%">Quantity</th>
                                <th width="20%">Total (₹)</th>
                                <th width="10%">Action</th>
                            </tr>
                        </thead>
                        <tbody id="products_tbody">
                            <!-- Initial row -->
                            <tr id="product_row_1">
                                <td>
                                    <select class="form-select product-select" name="products[0][product_id]" 
                                            data-row="1" onchange="fetchProductPrice(1)" required>
                                        <option value="">--SELECT PRODUCT--</option>
                                        @foreach($products as $product)
                                            <option value="{{ $product->id }}" 
                                                    data-price="{{ $product->price }}" 
                                                    data-quantity="{{ $product->quantity }}">
                                                {{ $product->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    <input type="hidden" name="products[0][product_name]" class="product-name">
                                </td>
                                <td>
                                    <input type="number" class="form-control price" name="products[0][price]" 
                                           id="price_1" readonly step="0.01">
                                </td>
                                <td>
                                    <input type="number" class="form-control quantity" name="products[0][quantity]" 
                                           id="quantity_1" min="1" value="1" onchange="calculateRowTotal(1)" required>
                                </td>
                                <td>
                                    <input type="number" class="form-control total" name="products[0][total]" 
                                           id="total_1" readonly step="0.01">
                                </td>
                                <td>
                                    <button type="button" class="btn btn-danger btn-sm" onclick="removeProductRow(1)">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </td>
                            </tr>
                        </tbody>
                        <tfoot>
                            <tr>
                                <td colspan="5">
                                    <button type="button" class="btn btn-primary btn-sm" onclick="addProductRow()">
                                        <i class="fas fa-plus"></i> Add Product
                                    </button>
                                </td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
            </div>
        </div>
        
        <!-- Form Actions -->
        <div class="row">
            <div class="col-md-12 text-center">
                <button type="submit" class="btn btn-success btn-lg">
                    <i class="fas fa-check"></i> Submit Invoice
                </button>
                <button type="button" class="btn btn-secondary btn-lg" onclick="resetForm()">
                    <i class="fas fa-redo"></i> Reset
                </button>
            </div>
        </div>
    </form>
</div>

<!-- JavaScript for Dynamic Functionality -->
<script>
    let rowCount = 1;
    
    // Add new product row
    function addProductRow() {
        rowCount++;
        const tbody = document.getElementById('products_tbody');
        const newRow = document.createElement('tr');
        newRow.id = `product_row_${rowCount}`;
        
        newRow.innerHTML = `
            <td>
                <select class="form-select product-select" name="products[${rowCount-1}][product_id]" 
                        data-row="${rowCount}" onchange="fetchProductPrice(${rowCount})" required>
                    <option value="">--SELECT PRODUCT--</option>
                    @foreach($products as $product)
                        <option value="{{ $product->id }}" 
                                data-price="{{ $product->price }}" 
                                data-quantity="{{ $product->quantity }}">
                            {{ $product->name }}
                        </option>
                    @endforeach
                </select>
                <input type="hidden" name="products[${rowCount-1}][product_name]" class="product-name">
            </td>
            <td>
                <input type="number" class="form-control price" name="products[${rowCount-1}][price]" 
                       id="price_${rowCount}" readonly step="0.01">
            </td>
            <td>
                <input type="number" class="form-control quantity" name="products[${rowCount-1}][quantity]" 
                       id="quantity_${rowCount}" min="1" value="1" onchange="calculateRowTotal(${rowCount})" required>
            </td>
            <td>
                <input type="number" class="form-control total" name="products[${rowCount-1}][total]" 
                       id="total_${rowCount}" readonly step="0.01">
            </td>
            <td>
                <button type="button" class="btn btn-danger btn-sm" onclick="removeProductRow(${rowCount})">
                    <i class="fas fa-trash"></i>
                </button>
            </td>
        `;
        
        tbody.appendChild(newRow);
    }
    
    // Remove product row
    function removeProductRow(rowId) {
        const row = document.getElementById(`product_row_${rowId}`);
        if (row && rowCount > 1) {
            row.remove();
            calculateInvoiceTotals();
        }
    }
    
    // Fetch product price
    function fetchProductPrice(rowId) {
        const selectElement = document.querySelector(`#product_row_${rowId} .product-select`);
        const selectedOption = selectElement.options[selectElement.selectedIndex];
        const priceInput = document.getElementById(`price_${rowId}`);
        const productNameInput = document.querySelector(`#product_row_${rowId} .product-name`);
        
        if (selectedOption.value) {
            const price = selectedOption.getAttribute('data-price');
            const productName = selectedOption.text;
            
            priceInput.value = price;
            productNameInput.value = productName;
            
            // Calculate row total
            calculateRowTotal(rowId);
        } else {
            priceInput.value = '';
            productNameInput.value = '';
            document.getElementById(`total_${rowId}`).value = '';
        }
    }
    
    // Calculate row total
    function calculateRowTotal(rowId) {
        const priceInput = document.getElementById(`price_${rowId}`);
        const quantityInput = document.getElementById(`quantity_${rowId}`);
        const totalInput = document.getElementById(`total_${rowId}`);
        
        const price = parseFloat(priceInput.value) || 0;
        const quantity = parseInt(quantityInput.value) || 0;
        
        const total = price * quantity;
        totalInput.value = total.toFixed(2);
        
        calculateInvoiceTotals();
    }
    
    // Calculate all invoice totals
    function calculateInvoiceTotals() {
        let subTotal = 0;
        const totalInputs = document.querySelectorAll('.total');
        
        totalInputs.forEach(input => {
            const value = parseFloat(input.value) || 0;
            subTotal += value;
        });
        
        const discount = parseFloat(document.getElementById('discount').value) || 0;
        const totalAmount = subTotal; // Same as sub_amount
        const gstAmount = (totalAmount - discount) * 0.18;
        const grandTotal = (totalAmount - discount) + gstAmount;
        
        // Update display fields
        document.getElementById('sub_amount_display').textContent = '₹' + subTotal.toFixed(2);
        document.getElementById('sub_amount').value = subTotal.toFixed(2);
        
        document.getElementById('total_amount_display').textContent = '₹' + totalAmount.toFixed(2);
        document.getElementById('total_amount').value = totalAmount.toFixed(2);
        
        document.getElementById('gst_amount_display').textContent = '₹' + gstAmount.toFixed(2);
        document.getElementById('gst_amount').value = gstAmount.toFixed(2);
        
        document.getElementById('grand_total_display').textContent = '₹' + grandTotal.toFixed(2);
        document.getElementById('grand_total').value = grandTotal.toFixed(2);
    }
    
    // Reset form
    function resetForm() {
        if (confirm('Are you sure you want to reset the form? All data will be lost.')) {
            document.getElementById('invoiceForm').reset();
            
            // Reset product rows to just one
            const tbody = document.getElementById('products_tbody');
            while (tbody.children.length > 1) {
                tbody.removeChild(tbody.lastChild);
            }
            
            // Reset the first row
            const firstRowSelect = document.querySelector('#product_row_1 .product-select');
            const firstRowPrice = document.getElementById('price_1');
            const firstRowQuantity = document.getElementById('quantity_1');
            const firstRowTotal = document.getElementById('total_1');
            
            firstRowSelect.value = '';
            firstRowPrice.value = '';
            firstRowQuantity.value = '1';
            firstRowTotal.value = '';
            
            rowCount = 1;
            calculateInvoiceTotals();
        }
    }
    
    // Initialize calculations
    document.addEventListener('DOMContentLoaded', function() {
        // Add event listener for discount
        document.getElementById('discount').addEventListener('input', calculateInvoiceTotals);
        
        // Calculate initial totals
        calculateInvoiceTotals();
    });
</script>

<style>
    .card {
        box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
        border: 1px solid #dee2e6;
    }
    
    .card-header {
        background-color: #f8f9fa;
        border-bottom: 1px solid #dee2e6;
    }
    
    .form-control:read-only {
        background-color: #f8f9fa;
    }
    
    #grand_total_display {
        background-color: #f0f9ff;
        font-size: 1.1rem;
    }
    
    .table th {
        background-color: #f8f9fa;
        font-weight: 600;
    }
    
    .btn-lg {
        min-width: 150px;
        margin: 0 10px;
    }
    
    .input-group .btn {
        border-top-left-radius: 0;
        border-bottom-left-radius: 0;
    }
</style>
@endsection