<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Quotation extends Model
{
    use HasFactory;

    protected $fillable = [
        'quotation_no',
        'customer_id',
        'quotation_date',
        'valid_till',
        'sub_amount',
        'discount',
        'gst_amount',
        'grand_total',
        'status',
        'notes'
    ];

    protected $casts = [
        'quotation_date' => 'date',
        'valid_till' => 'date',
        'sub_amount' => 'decimal:2',
        'discount' => 'decimal:2',
        'gst_amount' => 'decimal:2',
        'grand_total' => 'decimal:2',
    ];

    // IMPORTANT: Use 'approved' instead of 'accepted'
    protected $attributes = [
        'status' => 'draft',
    ];

    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    public function items()  // Make sure this matches your relationship name
    {
        return $this->hasMany(QuotationItem::class);
    }

    /**
     * Generate quotation number
     */
    public static function generateQuotationNo()
    {
        // Format: QTN-YYYYMM-001
        $prefix = 'QTN';
        $yearMonth = date('Ym');
        
        // Get the last quotation for this month
        $lastQuotation = self::where('quotation_no', 'like', "{$prefix}-{$yearMonth}-%")
            ->orderBy('id', 'desc')
            ->first();

        if ($lastQuotation) {
            // Extract the sequence number
            $lastNo = $lastQuotation->quotation_no;
            $parts = explode('-', $lastNo);
            $lastSequence = intval(end($parts));
            $sequence = str_pad($lastSequence + 1, 3, '0', STR_PAD_LEFT);
        } else {
            // First quotation of the month
            $sequence = '001';
        }

        return "{$prefix}-{$yearMonth}-{$sequence}";
    }
}