<?php
namespace App\Http\Controllers;

use App\Models\Quotation;
use App\Models\QuotationItem;
use App\Models\Customer;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class QuotationController extends Controller
{
    // Index - List all quotations
  public function index()
{
    $quotations = Quotation::with('customer')
        ->orderBy('created_at', 'desc')
        ->paginate(10);
    
    return view('quotation.index', compact('quotations'));
}


    // Create - Show create form
    public function create()
    {
        $customers = Customer::orderBy('name')->get();
        $products = Product::where('status', 'active')
            ->orderBy('name')
            ->get();
        
        return view('quotation.create', compact('customers', 'products'));
    }

    // Store - Save new quotation
  public function store(Request $request)
{
    // Validate request
    $validated = $request->validate([
        'customer_id' => 'required|exists:customers,id',
        'quotation_date' => 'required|date',
        'valid_till' => 'required|date|after_or_equal:quotation_date',
        'sub_amount' => 'required|numeric|min:0',
        'discount' => 'nullable|numeric|min:0',
        'gst_amount' => 'nullable|numeric|min:0',
        'grand_total' => 'required|numeric|min:0',
        'notes' => 'nullable|string',
        'products' => 'required|array|min:1',
        'products.*.product_id' => 'required|exists:products,id',
        'products.*.quantity' => 'required|integer|min:1',
        'products.*.price' => 'required|numeric|min:0',
        'products.*.subtotal' => 'required|numeric|min:0',
    ]);

    DB::beginTransaction();
    
    try {
        // Generate quotation number
        $quotationNo = $request->quotation_no;
        if (empty($quotationNo)) {
            // Use the method from the model
            $quotationNo = Quotation::generateQuotationNo();
        }
        
        // Create quotation
        $quotation = Quotation::create([
            'quotation_no' => $quotationNo,
            'customer_id' => $request->customer_id,
            'quotation_date' => $request->quotation_date,
            'valid_till' => $request->valid_till,
            'sub_amount' => $request->sub_amount,
            'discount' => $request->discount ?? 0,
            'gst_amount' => $request->gst_amount ?? 0,
            'grand_total' => $request->grand_total,
            'notes' => $request->notes,
            'status' => 'draft', // Default status
        ]);

        // Add quotation items
        foreach ($request->products as $productData) {
            QuotationItem::create([
                'quotation_id' => $quotation->id,
                'product_id' => $productData['product_id'],
                'quantity' => $productData['quantity'],
                'price' => $productData['price'],
                'subtotal' => $productData['subtotal'],
            ]);

            // Optional: Update product stock if you want to reserve items
            // $product = Product::find($productData['product_id']);
            // if ($product) {
            //     $product->decrement('quantity', $productData['quantity']);
            // }
        }

        DB::commit();

        return redirect()->route('quotation.show', $quotation->id)
            ->with('success', 'Quotation created successfully!');

    } catch (\Exception $e) {
        DB::rollBack();
        return back()->withInput()
            ->with('error', 'Failed to create quotation. Error: ' . $e->getMessage());
    }
}

    // Show - Display single quotation
    public function show(Quotation $quotation)
    {
        $quotation->load(['customer', 'items.product']);
        return view('quotation.show', compact('quotation'));
    }

    // Edit - Show edit form
   public function edit(Quotation $quotation)
{
    $customers = Customer::all();
    $products = Product::all();
    
    // Get existing quotation items
    $existingProducts = $quotation->quotationItems()->with('product')->get();
    
    // Prepare products data for JavaScript
    $productsData = [];
    foreach ($existingProducts as $key => $item) {
        $productsData[] = [
            'product_id' => $item->product_id,
            'quantity' => $item->quantity,
            'price' => $item->price,
            'subtotal' => $item->subtotal
        ];
    }
    
    return view('quotation.edit', compact('quotation', 'customers', 'products', 'productsData'));
}

    // Update - Update quotation
  public function update(Request $request, Quotation $quotation)
{
    $validated = $request->validate([
        'customer_id' => 'required|exists:customers,id',
        'quotation_date' => 'required|date',
        'valid_till' => 'required|date|after_or_equal:quotation_date',
        'sub_amount' => 'required|numeric|min:0',
        'discount' => 'nullable|numeric|min:0',
        'gst_amount' => 'nullable|numeric|min:0',
        'grand_total' => 'required|numeric|min:0',
        'notes' => 'nullable|string',
        'status' => 'nullable|in:draft,sent,accepted,rejected,expired',
        'products' => 'required|array|min:1',
        'products.*.product_id' => 'required|exists:products,id',
        'products.*.quantity' => 'required|integer|min:1',
        'products.*.price' => 'required|numeric|min:0',
        'products.*.subtotal' => 'required|numeric|min:0',
    ]);

    DB::beginTransaction();

    try {
        // Update quotation
        $quotation->update([
            'customer_id' => $request->customer_id,
            'quotation_date' => $request->quotation_date,
            'valid_till' => $request->valid_till,
            'sub_amount' => $request->sub_amount,
            'discount' => $request->discount ?? 0,
            'gst_amount' => $request->gst_amount ?? 0,
            'grand_total' => $request->grand_total,
            'notes' => $request->notes,
            'status' => $request->status ?? $quotation->status,
        ]);

        // Delete existing items
        $quotation->quotationItems()->delete();

        // Add new items
        foreach ($request->products as $productData) {
            QuotationItem::create([
                'quotation_id' => $quotation->id,
                'product_id' => $productData['product_id'],
                'quantity' => $productData['quantity'],
                'price' => $productData['price'],
                'subtotal' => $productData['subtotal'],
            ]);
        }

        DB::commit();

        return redirect()->route('quotation.show', $quotation->id)
            ->with('success', 'Quotation updated successfully!');

    } catch (\Exception $e) {
        DB::rollBack();
        return back()->withInput()
            ->with('error', 'Failed to update quotation. Error: ' . $e->getMessage());
    }
}

    // Destroy - Delete quotation
    public function destroy(Quotation $quotation)
    {
        DB::beginTransaction();
        
        try {
            $quotation->items()->delete();
            $quotation->delete();
            
            DB::commit();
            
            return redirect()->route('quotation.index')
                ->with('success', 'Quotation deleted successfully!');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to delete quotation. Error: ' . $e->getMessage());
        }
    }

    // Print/Download PDF
    public function print(Quotation $quotation)
    {
        $quotation->load(['customer', 'items.product']);
        return view('quotation.print', compact('quotation'));
    }
}