<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category; // Add this
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ProductController extends Controller
{
    // Show all products
    public function index() {
        $products = Product::with('category')->latest()->get(); // Add with('category')
        return view('products.index', compact('products'));
    }

    // Show form to create product
    public function create() {
        $categories = Category::orderBy('name')->get(); // Get categories from DB
        return view('products.create', compact('categories')); // Pass categories to view
    }

    // Store new product
    public function store(Request $request)
{
    $request->validate([
        'name' => 'required|string|max:255',
        'description' => 'nullable|string',
        'sku' => 'nullable|string|unique:products,sku',
        'category_id' => 'required|exists:categories,id',
        'price' => 'required|numeric|min:0',
        'quantity' => 'required|integer|min:0',
        'unit' => 'nullable|string|max:50',
        'status' => 'required|in:active,inactive,out_of_stock',
        'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
    ]);

    $data = $request->only(
        'name','description','sku','price','quantity','unit','status'
    );

    $data['category_id'] = $request->category_id;

    // ✅ image upload (CORRECT WAY)
    if ($request->hasFile('image')) {

        $image = $request->file('image');
        $filename = time() . '_' . $image->getClientOriginalName();

        $image->move(public_path('uploads/products'), $filename);

        // ✅ save into $data (NOT $product)
        $data['image'] = 'uploads/products/' . $filename;
    }

    // ✅ now create product
    Product::create($data);

    return redirect()->route('products.index')
        ->with('success', 'Product added successfully!');
}


    // Show form to edit product
    public function edit(Product $product) {
        $categories = Category::orderBy('name')->get(); // Get categories from DB
        return view('products.edit', compact('product', 'categories')); // Pass categories
    }

    // Update product
    public function update(Request $request, Product $product) {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'sku' => 'nullable|string|unique:products,sku,' . $product->id, // Unique except current
            'category_id' => 'required|exists:categories,id', // Change to category_id
            'price' => 'required|numeric|min:0',
            'quantity' => 'required|integer|min:0',
            'unit' => 'nullable|string|max:50',
            'status' => 'required|in:active,inactive,out_of_stock',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ]);

        $data = $request->only('name','description','sku','price','quantity','unit','status');
        $data['category_id'] = $request->category_id; // Store category_id

        if ($request->hasFile('image')) {

    // delete old image
    if ($product->image && file_exists(public_path($product->image))) {
        unlink(public_path($product->image));
    }

    $image = $request->file('image');
    $filename = time() . '_' . $image->getClientOriginalName();
    $image->move(public_path('uploads/products'), $filename);

    $data['image'] = 'uploads/products/' . $filename;
}


        $product->update($data);

        return redirect()->route('products.index')->with('success', 'Product updated successfully!');
    }

    // Delete product
    public function destroy(Product $product)
{
    if ($product->image && file_exists(public_path($product->image))) {
        unlink(public_path($product->image));
    }

    $product->delete();

    return redirect()->route('products.index')
        ->with('success', 'Product deleted successfully!');
}


    // View single product
    public function show(Product $product) {
        return view('products.show', compact('product'));
    }
}