<?php

namespace App\Http\Controllers;

use App\Models\InvoiceItem;
use App\Models\Invoice;
use App\Models\Product;
use Illuminate\Http\Request;

class InvoiceItemController extends Controller
{
    // Show all items for a specific invoice
    public function index($invoiceId)
    {
        $invoice = Invoice::with('items.product')->findOrFail($invoiceId);
        return view('invoice_items.index', compact('invoice'));
    }

    // Show form to add a new item to invoice
    public function create($invoiceId)
    {
        $invoice = Invoice::findOrFail($invoiceId);
        $products = Product::all();
        return view('invoice_items.create', compact('invoice', 'products'));
    }

    // Store new invoice item
    public function store(Request $request, $invoiceId)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
        ]);

        $invoice = Invoice::findOrFail($invoiceId);
        $product = Product::findOrFail($request->product_id);

        $subtotal = $product->price * $request->quantity;

        InvoiceItem::create([
            'invoice_id' => $invoice->id,
            'product_id' => $product->id,
            'quantity' => $request->quantity,
            'price' => $product->price,
            'subtotal' => $subtotal,
        ]);

        // Update invoice total
        $invoice->total += $subtotal;
        $invoice->save();

        return redirect()->route('invoice_items.index', $invoiceId)
                         ->with('success', 'Item added successfully!');
    }

    // Show edit form for invoice item
    public function edit($id)
    {
        $item = InvoiceItem::findOrFail($id);
        $products = Product::all();
        return view('invoice_items.edit', compact('item', 'products'));
    }

    // Update invoice item
    public function update(Request $request, $id)
    {
        $item = InvoiceItem::findOrFail($id);

        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
        ]);

        $oldSubtotal = $item->subtotal;

        $product = Product::findOrFail($request->product_id);
        $item->update([
            'product_id' => $product->id,
            'quantity' => $request->quantity,
            'price' => $product->price,
            'subtotal' => $product->price * $request->quantity,
        ]);

        // Update invoice total
        $invoice = $item->invoice;
        $invoice->total = $invoice->total - $oldSubtotal + $item->subtotal;
        $invoice->save();

        return redirect()->route('invoice_items.index', $invoice->id)
                         ->with('success', 'Item updated successfully!');
    }

    // Delete invoice item
    public function destroy($id)
    {
        $item = InvoiceItem::findOrFail($id);
        $invoice = $item->invoice;

        $invoice->total -= $item->subtotal;
        $invoice->save();

        $item->delete();

        return redirect()->route('invoice_items.index', $invoice->id)
                         ->with('success', 'Item deleted successfully!');
    }
}
