<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Customer;
use App\Models\Product;
use Illuminate\Http\Request;
use DB;

class InvoiceController extends Controller
{
    public function index()
    {
        $invoices = Invoice::with('customer')->latest()->get();
        return view('invoices.index', compact('invoices'));
    }

    public function create()
    {
        $customers = Customer::all();
        $products = Product::all();

        return view('invoices.create', compact('customers', 'products'));
    }

    public function store(Request $request)
    {
        // Validation
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'invoice_date' => 'required|date',
            'invoice_no' => 'required|unique:invoices,invoice_no',
            'sub_amount' => 'required|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'total_amount' => 'required|numeric|min:0',
            'gst_amount' => 'required|numeric|min:0',
            'grand_total' => 'required|numeric|min:0',
            'products' => 'required|array|min:1',
            'products.*.product_id' => 'required|exists:products,id',
            'products.*.quantity' => 'required|integer|min:1',
        ]);

        DB::beginTransaction();

        try {
            // Check product availability
            foreach ($request->products as $product) {
                $productModel = Product::find($product['product_id']);
                if (!$productModel) {
                    return back()->withErrors(['products' => 'Product not found'])->withInput();
                }
                if ($productModel->quantity < $product['quantity']) {
                    return back()
                        ->withErrors(['products' => "Insufficient stock for {$productModel->name}. Available: {$productModel->quantity}"])
                        ->withInput();
                }
            }

            // Create invoice
            $invoice = Invoice::create([
                'invoice_no' => $request->invoice_no,
                'customer_id' => $request->customer_id,
                'invoice_date' => $request->invoice_date,
                'sub_amount' => $request->sub_amount,
                'discount' => $request->discount ?? 0,
                'total_amount' => $request->total_amount,
                'gst_amount' => $request->gst_amount,
                'grand_total' => $request->grand_total,
                'paid_amount' => 0,
                'due_amount' => $request->grand_total,
                'payment_type' => 'cash',
                'payment_status' => 'unpaid',
                'payment_place' => 'store',
            ]);

            // Create invoice items
            foreach ($request->products as $product) {
                $productModel = Product::find($product['product_id']);
                
                InvoiceItem::create([
                    'invoice_id' => $invoice->id,
                    'product_id' => $product['product_id'],
                    'quantity' => $product['quantity'],
                    'price' => $product['price'],
                    'subtotal' => $product['total'],
                ]);

                // Reduce stock
                $productModel->decrement('quantity', $product['quantity']);
            }

            DB::commit();

            return redirect()
                ->route('invoices.index')
                ->with('success', 'Invoice created successfully!');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error creating invoice: ' . $e->getMessage());
        }
    }

    public function show(Invoice $invoice)
    {
        $invoice->load('customer', 'items.product');
        return view('invoices.show', compact('invoice'));
    }

   // app/Http/Controllers/InvoiceController.php

public function edit($id)
{
    $invoice = Invoice::with(['customer', 'invoiceItems.product'])->findOrFail($id);
    $customers = Customer::all();
    $products = Product::all();
    
    return view('invoices.edit', compact('invoice', 'customers', 'products'));
}

   public function update(Request $request, Invoice $invoice)
{
    // Validation
    $request->validate([
        'customer_id' => 'required|exists:customers,id',
        'invoice_date' => 'required|date',
        'sub_amount' => 'required|numeric|min:0',
        'discount' => 'nullable|numeric|min:0',
        'total_amount' => 'required|numeric|min:0',
        'gst_amount' => 'required|numeric|min:0',
        'grand_total' => 'required|numeric|min:0',
        'products' => 'required|array|min:1',
        'products.*.product_id' => 'required|exists:products,id',
        'products.*.quantity' => 'required|integer|min:1',
        'products.*.price' => 'required|numeric|min:0',
        'products.*.total' => 'required|numeric|min:0',
    ]);

    DB::beginTransaction();

    try {
        // Step 1: Prepare arrays to track old and new quantities
        $oldProducts = [];
        $newProducts = [];
        
        // Get old invoice items quantities
        foreach ($invoice->items as $item) {
            $oldProducts[$item->product_id] = $item->quantity;
        }
        
        // Get new product quantities
        foreach ($request->products as $productData) {
            $productId = $productData['product_id'];
            $newProducts[$productId] = ($newProducts[$productId] ?? 0) + $productData['quantity'];
        }

        // Step 2: Check stock availability for all products
        foreach ($newProducts as $productId => $newQuantity) {
            $product = Product::find($productId);
            
            if (!$product) {
                throw new \Exception("Product ID {$productId} not found.");
            }
            
            // Calculate available stock considering old quantities being returned
            $oldQuantity = $oldProducts[$productId] ?? 0;
            $currentStock = $product->quantity;
            
            // If this product was in old invoice, we'll return it to stock
            // So available stock = current stock + old quantity
            $availableStock = $currentStock + $oldQuantity;
            
            if ($newQuantity > $availableStock) {
                throw new \Exception("Insufficient stock for {$product->name}. Available: {$availableStock}, Requested: {$newQuantity}");
            }
        }

        // Step 3: Return old quantities to stock first
        foreach ($oldProducts as $productId => $oldQuantity) {
            $product = Product::find($productId);
            if ($product) {
                $product->increment('quantity', $oldQuantity);
            }
        }

        // Step 4: Update invoice details
        $invoice->update([
            'customer_id' => $request->customer_id,
            'invoice_date' => $request->invoice_date,
            'sub_amount' => $request->sub_amount,
            'discount' => $request->discount ?? 0,
            'total_amount' => $request->total_amount,
            'gst_amount' => $request->gst_amount,
            'grand_total' => $request->grand_total,
            'due_amount' => $request->grand_total - $invoice->paid_amount, // Recalculate due amount
        ]);

        // Step 5: Delete old invoice items
        $invoice->items()->delete();

        // Step 6: Create new invoice items and update stock
        foreach ($request->products as $productData) {
            $product = Product::find($productData['product_id']);
            
            // Create new invoice item
            InvoiceItem::create([
                'invoice_id' => $invoice->id,
                'product_id' => $productData['product_id'],
                'quantity' => $productData['quantity'],
                'price' => $productData['price'],
                'subtotal' => $productData['total'],
            ]);

            // Reduce stock with new quantity
            $product->decrement('quantity', $productData['quantity']);
        }

        DB::commit();

        return redirect()
            ->route('invoices.index')
            ->with('success', 'Invoice updated successfully!');

    } catch (\Exception $e) {
        DB::rollBack();
        
        return back()
            ->withErrors(['error' => 'Error updating invoice: ' . $e->getMessage()])
            ->withInput();
    }
}
    public function destroy(Invoice $invoice)
    {
        DB::beginTransaction();
        
        try {
            // Restore product quantities
            foreach ($invoice->items as $item) {
                $product = Product::find($item->product_id);
                if ($product) {
                    $product->increment('quantity', $item->quantity);
                }
            }
            
            // Delete invoice items
            $invoice->items()->delete();
            
            // Delete invoice
            $invoice->delete();
            
            DB::commit();
            
            return redirect()
                ->route('invoices.index')
                ->with('success', 'Invoice deleted successfully!');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error deleting invoice: ' . $e->getMessage());
        }
    }

    // public function download(Invoice $invoice)
    // {
    //     $invoice->load('customer', 'items.product');
        
    //     // For now, return a view. You can implement PDF later.
    //     return view('invoices.download', compact('invoice'));
        
    //     // If you want PDF, install: composer require barryvdh/laravel-dompdf
    //     // Then uncomment:
    //     /*
    //     $pdf = \PDF::loadView('invoices.pdf', compact('invoice'));
    //     return $pdf->download('invoice-' . $invoice->invoice_no . '.pdf');
    //     */
    // }


  public function download($invoice_id)
{
    // Fetch invoice details
    $invoice = Invoice::findOrFail($invoice_id);

    // Fetch all products directly
    $products = Product::all(); // Or filter for invoice if needed

    // Pass to Blade
    return view('invoices.download', compact('invoice', 'products'));
}

}